"use client"

import { notFound } from "next/navigation"
import Image from "next/image"
import { 
  Building2, 
  Globe, 
  MapPin, 
  PenSquare, 
  Share, 
  Trash2, 
  Users 
} from "lucide-react"

import { companies, contacts, deals } from "@/lib/mock-data"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"

interface CompanyDetailsPageProps {
  params: {
    id: string
  }
}

export default function CompanyDetailsPage({ params }: CompanyDetailsPageProps) {
  // Find the company with the matching ID
  const company = companies.find(c => c.id === params.id)

  // If company not found, show 404
  if (!company) {
    notFound()
  }

  // Get contacts for this company
  const companyContacts = contacts.filter(contact => contact.company === company.name)
  
  // Get deals for this company
  const companyDeals = deals.filter(deal => deal.company === company.name)

  return (
    <div className="p-6 space-y-8">
      <div className="flex justify-between items-start">
        <div className="space-y-1">
          <h2 className="text-3xl font-bold tracking-tight">{company.name}</h2>
          {company.industry && (
            <p className="text-muted-foreground">{company.industry}</p>
          )}
        </div>
        <div className="flex space-x-2">
          <Button variant="outline" size="sm">
            <Share className="h-4 w-4 mr-2" />
            Share
          </Button>
          <Button variant="outline" size="sm">
            <PenSquare className="h-4 w-4 mr-2" />
            Edit
          </Button>
          <Button variant="outline" size="sm" className="text-destructive hover:text-destructive">
            <Trash2 className="h-4 w-4 mr-2" />
            Delete
          </Button>
        </div>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        {/* Company information */}
        <div className="lg:col-span-1 space-y-6">
          <Card>
            <CardHeader>
              <CardTitle>Company Details</CardTitle>
            </CardHeader>
            <CardContent className="space-y-6">
              {company.logo && (
                <div className="flex justify-center">
                  <div className="h-32 w-32 rounded-md overflow-hidden relative">
                    <Image
                      src={company.logo}
                      alt={company.name}
                      fill
                      className="object-cover"
                    />
                  </div>
                </div>
              )}
              
              <div className="space-y-3">
                {company.website && (
                  <div className="flex items-center text-sm">
                    <Globe className="h-4 w-4 mr-2 text-muted-foreground" />
                    <a 
                      href={company.website} 
                      target="_blank" 
                      rel="noopener noreferrer"
                      className="text-blue-600 dark:text-blue-400 hover:underline"
                    >
                      {company.website.replace(/^https?:\/\//, '')}
                    </a>
                  </div>
                )}
                
                {company.location && (
                  <div className="flex items-center text-sm">
                    <MapPin className="h-4 w-4 mr-2 text-muted-foreground" />
                    <span>{company.location}</span>
                  </div>
                )}
                
                {company.size && (
                  <div className="flex items-center text-sm">
                    <Users className="h-4 w-4 mr-2 text-muted-foreground" />
                    <span>Size: {company.size}</span>
                  </div>
                )}
                
                <div className="flex items-center text-sm">
                  <Building2 className="h-4 w-4 mr-2 text-muted-foreground" />
                  <span>{companyContacts.length} contacts</span>
                </div>
              </div>
              
              <div>
                <h3 className="text-sm font-medium mb-2">Notes</h3>
                <p className="text-sm text-muted-foreground">
                  {company.notes || "No notes available for this company."}
                </p>
              </div>
            </CardContent>
          </Card>
        </div>
        
        {/* Contacts and Deals */}
        <div className="lg:col-span-2 space-y-6">
          {/* Contacts */}
          <Card>
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle>Contacts</CardTitle>
              <Button variant="outline" size="sm">
                Add Contact
              </Button>
            </CardHeader>
            <CardContent>
              {companyContacts.length === 0 ? (
                <p className="text-muted-foreground text-sm">No contacts for this company.</p>
              ) : (
                <div className="space-y-4">
                  {companyContacts.map(contact => (
                    <div key={contact.id} className="flex items-center space-x-3 border-b pb-3 last:border-0">
                      {contact.avatar ? (
                        <div className="h-10 w-10 rounded-full overflow-hidden relative">
                          <Image
                            src={contact.avatar}
                            alt={contact.name}
                            fill
                            className="object-cover"
                          />
                        </div>
                      ) : (
                        <div className="h-10 w-10 rounded-full bg-muted flex items-center justify-center">
                          <Users className="h-5 w-5 text-muted-foreground" />
                        </div>
                      )}
                      <div className="flex-1 min-w-0">
                        <p className="font-medium">{contact.name}</p>
                        <p className="text-sm text-muted-foreground">{contact.position}</p>
                      </div>
                      <div className="text-sm">
                        <a 
                          href={`mailto:${contact.email}`} 
                          className="text-blue-600 dark:text-blue-400 hover:underline"
                        >
                          {contact.email}
                        </a>
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
          
          {/* Deals */}
          <Card>
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle>Deals</CardTitle>
              <Button variant="outline" size="sm">
                Add Deal
              </Button>
            </CardHeader>
            <CardContent>
              {companyDeals.length === 0 ? (
                <p className="text-muted-foreground text-sm">No deals for this company.</p>
              ) : (
                <div className="space-y-4">
                  {companyDeals.map(deal => (
                    <div key={deal.id} className="flex items-center justify-between border-b pb-3 last:border-0">
                      <div className="flex-1 min-w-0">
                        <p className="font-medium">{deal.title}</p>
                        <p className="text-sm text-muted-foreground">Contact: {deal.contact}</p>
                      </div>
                      <div className="flex flex-col items-end">
                        <span className="font-medium">{deal.value}</span>
                        <span className={`text-xs px-2 py-1 rounded-full 
                          ${deal.stage === 'qualified' ? 'bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-300' :
                            deal.stage === 'proposal' ? 'bg-amber-100 text-amber-800 dark:bg-amber-900 dark:text-amber-300' :
                            deal.stage === 'negotiation' ? 'bg-violet-100 text-violet-800 dark:bg-violet-900 dark:text-violet-300' :
                            'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300'
                          }`}
                        >
                          {deal.stage.charAt(0).toUpperCase() + deal.stage.slice(1)}
                        </span>
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  )
} 